/*
   hdmotion - Moves hard disk heads in interesting patterns
   (C) 2005 by Jeremy Stanley

   This program may be distributed freely provided this notice
	 is preserved.

   The author assumes no liability for any damages arising
   out of the use of this program, including but not limited
   to loss of data or desire to open up operational hard drives.
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dos.h>
#include <time.h>
#include <conio.h>
#include <math.h>

typedef int BOOL;
#define FALSE 0
#define TRUE (!FALSE)

/* -------------------------
   Extended INT 13H routines
   ------------------------- */

void print_error(unsigned char code)
{
  const char *message;

  switch(code)
  {
    case 0x01: message = "Invalid function"; break;
    case 0x02: message = "Address mark not found"; break;
    case 0x03: message = "Disk write protected"; break;
    case 0x04: message = "Sector not found"; break;
    case 0x07: message = "Drive parameter activity failed"; break;
    case 0x08: message = "DMA overrun"; break;
    case 0x09: message = "Data boundary error"; break;
    case 0x0A: message = "Bad sector detected"; break;
    case 0x0B: message = "Bad track detected"; break;
    case 0x0C: message = "Unsupported track or invalid media"; break;
    case 0x0E: message = "Control data address mark detected"; break;
    case 0x0F: message = "DMA arbitration level out of range"; break;
    case 0x10: message = "Uncorrectable CRC or ECC error on read"; break;
    case 0x11: message = "Data ECC corrected"; break;
    case 0x31: message = "No media in drive"; break;
    case 0x80: case 0xAA: message = "Drive not ready"; break;
    case 0xB3: message = "Volume in use"; break;
    case 0xCC: message = "Write fault"; break;
		case 0xE0: message = "Status register error"; break;
    case 0xFF: message = "Sense operation failed"; break;
    default: message = "Failure on INT 13";
  }

  fprintf(stderr, "%02x %s\n", code, message);
}

/* Test for INT 13 Extensions support */
BOOL support_13x(unsigned char drv_num)
{
  unsigned char version;
  unsigned short flags;

  asm {
    mov ah, 0x41
    mov bx, 0x55AA
    mov dl, drv_num
    int 0x13
    jc error_0
    cmp bx, 0xAA55
    jnz error_0
    mov version, ah
    mov flags, cx
  }

  if ( flags & 1 )
  {
    return TRUE;
  }
  else
  {
    return FALSE;
  }

error_0:
  return FALSE;
}

/* Get the size and count of sectors on the drive */
BOOL get_drive_parameters(unsigned char drv_num, unsigned short *bytes_per_sector, unsigned long *sectors)
{
  const static size_t bufsize = 0x1E;

  char parm_buf[bufsize];
  char far *pBuf = parm_buf;
  unsigned char error;

  memset(pBuf, 0, bufsize);
  *(unsigned short *)pBuf = bufsize;

  asm {
    mov ah,0x48
    mov dl,drv_num
    push ds
    lds si,pBuf
    int 0x13
    pop ds
    jc error_1
  }

  *bytes_per_sector = *(unsigned short *)(pBuf + 0x18);
  *sectors = *(unsigned long *)(pBuf + 0x10);

  return TRUE;

error_1:
  asm {
    mov error,ah
  }
	print_error(error);

  return FALSE;
}

/* Transfer sectors */
BOOL xfer_sectors(unsigned char drv_num, BOOL write, unsigned long sector, unsigned short count, void far *buf)
{

  char addx[16];
  char far *pAddx = addx;
  unsigned char error;

  unsigned char op = (write ? 0x43 : 0x42);

  *(unsigned char *)(addx) = 16;
  *(unsigned char *)(addx + 1) = 0;
  *(unsigned short *)(addx + 2) = count;
  *(void far **)(addx + 4) = buf;
  *(unsigned long *)(addx + 8) = sector;
  *(unsigned long *)(addx + 12) = 0L;

  asm {
    mov ah, op
    mov al, 0
    mov dl, drv_num
    push ds
    lds si, pAddx
    int 0x13
    pop ds
    jc error_2
  }

  return TRUE;

error_2:
  asm {
    mov error, ah
  }
  print_error(error);
  return FALSE;
}

/* -------------------------
   vatchen das blinkenlights
   ------------------------- */

class hdmotion
{
public:
  hdmotion() :
  	drvnum(0),
    bytes_per_sector(0),
    capacity(0),
    buf_sectors(0),
    bufsize(0),
    buf(NULL)
  {
  }

  ~hdmotion()
  {
  	if ( buf )
    	delete [] buf;
  }

	BOOL open_drive(unsigned char drvnum_)
  {
  	drvnum = drvnum_;

	  if ( !support_13x(drvnum) )
	  {
	  	fprintf(stderr, "Drive not present or INT 13 Extensions not supported\n");
	    return FALSE;
	  }

  	if ( !get_drive_parameters(drvnum, &bytes_per_sector, &capacity) )
	  {
	    fprintf(stderr, "Failed to query drive capacity\n");
	  	return FALSE;
	  }

		buf_sectors = 65535 / bytes_per_sector;
    bufsize = buf_sectors * bytes_per_sector;
    buf = new unsigned char[bufsize];
    if ( !buf )
    {
    	fprintf(stderr, "Failed to allocate sector buffer.  DOS sucks.\n");
      return FALSE;
    }

    set_chunk(1);

    return TRUE;
  }

  void set_chunk(unsigned int _c)
  {
  	chunk = _c;
    if ( chunk > buf_sectors )
	    chunk = buf_sectors;
  }

  void move_head(double position)
  {
    static char linebuf[80];
    memset(linebuf, ' ', 79);
    linebuf[(int)(position * 79)] = '#';
    linebuf[79] = '\0';
		puts(linebuf);

    unsigned long start_sec = (unsigned long)(position * (capacity - chunk));
		xfer_sectors(drvnum, FALSE, start_sec, chunk, buf);

		if ( kbhit() && (getch() == 27) )
    	exit(0);
  }

private:
	unsigned char drvnum;
  unsigned short bytes_per_sector;
  unsigned long capacity;
  unsigned int buf_sectors;
  unsigned int chunk;
  size_t bufsize;
  unsigned char *buf;
};

/* -------------------------
   main()
   ------------------------- */


int main(int argc, char **argv)
{
  if (argc < 2)
  {
  	fprintf(stderr, "Please specify a drive number (0x80, 0x81, etc...)\n");
    return 1;
  }

	unsigned char drvnum = (unsigned char)strtoul(argv[1], NULL, 0);
  hdmotion D;
  if ( !D.open_drive(drvnum) )
  	return 1;


  double f, s, l, h, amp;
  int i, heads;

  // accelerating zigzag
  s = 0.010;
  for(i = 0; i < 5; ++i)
  {
		for(f = 0.0; f < 1.0; f += s)
	  	D.move_head(f);
	  for(f -= s; f > 0.0; f -= s)
	  	D.move_head(f);

    s += 0.0075;
  }
  f += s;

  // tightening zigzag
  h = 0.90;
  l = 0.10;
  for(; l < h;)
  {
  	for(; f < h; f += s)
    	D.move_head(f);
    for(; f > l; f -= s)
    	D.move_head(f);

    h -= 0.05;
    l += 0.05;
  }

  // widening sinusoid
  amp = 0.05;
  for(; amp <= 0.50; amp += 0.05)
  {
    double x;
  	for(x = 0; x < (2 * M_PI); x += M_PI / 32.0)
    {
    	D.move_head( (sin(x) * amp) + 0.5 );
    }
  }

  // narrowing sinusoid
  for(amp = 0.50; amp > 0.0; amp -= 0.05)
  {
    double x;
  	for(x = 0; x < (2 * M_PI); x += M_PI / 32.0)
    {
    	D.move_head( (sin(x) * amp) + 0.5 );
    }
  }

  // widening double-sinusoid
  amp = 0.05;
  for(; amp <= 0.50; amp += 0.05)
  {
    double x;
  	for(x = 0; x < (2 * M_PI); x += M_PI / 16.0)
    {
    	f = (sin(x) * amp) + 0.5;
    	D.move_head( f );
      D.move_head( 1.0 - f );
    }
  }

  // narrowing double-sinusoid
  for(amp = 0.50; amp > 0.0; amp -= 0.05)
  {
    double x;
  	for(x = 0; x < (2 * M_PI); x += M_PI / 16.0)
    {
    	f = (sin(x) * amp) + 0.5;
    	D.move_head( f );
      D.move_head( 1.0 - f );
		}
  }

  // buncha heads
	for(heads = 2; heads < 7; ++heads)
	{
    int repeat = 160 / heads;
	  for(int i = 0; i < repeat; ++i)
	  {
    	for(int j = 1; j <= heads; ++j)
      {
		  	D.move_head((double)j / (heads + 1));
	    }
	  }
  }
  for(; heads > 0; heads -= 2)
	{
    int repeat = 160 / heads;
	  for(int i = 0; i < repeat; ++i)
	  {
    	for(int j = 1; j <= heads; ++j)
      {
		  	D.move_head((double)j / (heads + 1));
	    }
	  }
  }

  // noise
  for(i = 0; i < 600; ++i)
  {
  	D.move_head((double)rand() / RAND_MAX);
  }

  return 0;
}
